// rdgpimon.cpp
//
// A Qt-based application for testing General Purpose Input (GPI) devices.
//
//   (C) Copyright 2002-2003 Fred Gleason <fredg@paravelsystems.com>
//
//      $Id: rdgpimon.cpp,v 1.4 2007/02/14 21:59:12 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License version 2 as
//   published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//


#include <qapplication.h>
#include <qwidget.h>
#include <qpushbutton.h>
#include <qrect.h>
#include <qpoint.h>
#include <qpainter.h>
#include <qstring.h>
#include <qmessagebox.h>
#include <qlineedit.h>
#include <qtextcodec.h>
#include <qtranslator.h>

#include <rconf.h>

#include <rdgpimon.h>
#include <rdcmd_switch.h>
#include <rddbheartbeat.h>

//
// Icons
//
#include "../icons/rivendell-22x22.xpm"


MainWidget::MainWidget(QWidget *parent,const char *name)
  :QWidget(parent,name)
{
  unsigned mask=0;
  QString str;

  gpi_grid_height=0;

  //
  // HACK: Disable the Broken Custom SuSE Dialogs
  //
  setenv("QT_NO_KDE_INTEGRATION","1",1);

  //
  // Read Command Options
  //
  RDCmdSwitch *cmd=new RDCmdSwitch(qApp->argc(),qApp->argv(),"rdgpimon","\n");
  delete cmd;

  //
  // Generate Font
  //
  QFont font("Helvetica",12,QFont::Bold);
  font.setPixelSize(12);
  QFont label_font("Helvetica",18,QFont::Bold);
  label_font.setPixelSize(18);

  setCaption("RDGpiMon");
  for(int i=0;i<MAX_GPIO_PINS;i++) {
    gpi_state[i]=false;
  }

  //
  // Create And Set Icon
  //
  gpi_rivendell_map=new QPixmap(rivendell_xpm);
  setIcon(*gpi_rivendell_map);

  //
  // Load Local Configs
  //
  gpi_config=new RDConfig();
  gpi_config->load();

  str=QString(tr("RDAirPlay - Station:"));
  setCaption(QString().sprintf("%s %s",(const char *)str,
			       (const char *)gpi_config->stationName()));

  //
  // Open Database
  //
  gpi_db=QSqlDatabase::addDatabase(gpi_config->mysqlDriver());
  if(!gpi_db) {
    QMessageBox::warning(this,tr("Database Error"),
		    tr("Can't Connect","Unable to connect to mySQL Server!"));
    exit(0);
  }
  gpi_db->setDatabaseName(gpi_config->mysqlDbname());
  gpi_db->setUserName(gpi_config->mysqlUsername());
  gpi_db->setPassword(gpi_config->mysqlPassword());
  gpi_db->setHostName(gpi_config->mysqlHostname());
  if(!gpi_db->open()) {
    QMessageBox::warning(this,tr("Can't Connect"),
			 tr("Unable to connect to mySQL Server!"));
    gpi_db->removeDatabase(gpi_config->mysqlDbname());
    exit(0);
  }
  new RDDbHeartbeat(gpi_config->mysqlHeartbeatInterval(),this);

  //
  // RIPC Connection
  //
  gpi_ripc=new RDRipc(gpi_config->stationName());
  connect(gpi_ripc,SIGNAL(userChanged()),this,SLOT(userData()));
  connect(gpi_ripc,SIGNAL(gpiStateChanged(int,int,bool)),
	  this,SLOT(gpiStateChangedData(int,int,bool)));
  connect(gpi_ripc,SIGNAL(gpiMaskChanged(int,int,bool)),
	  this,SLOT(gpiMaskChangedData(int,int,bool)));
  connect(gpi_ripc,SIGNAL(gpiCartChanged(int,int,int)),
	  this,SLOT(gpiCartChangedData(int,int,int)));
  gpi_ripc->connectHost("localhost",RIPCD_TCP_PORT,gpi_config->password());

  //
  // RDStation
  //
  gpi_station=new RDStation(gpi_config->stationName());

  //
  // RDMatrix;
  //
  gpi_matrix=NULL;

  //
  // Matrix Selector
  //
  gpi_matrix_box=new QComboBox(this,"gpi_matrix_box");
  gpi_matrix_box->setGeometry(80,10,80,21);
  for(int i=0;i<MAX_MATRICES;i++) {
    gpi_matrix_box->insertItem(QString().sprintf("%d",i));
  }
  QLabel *label=new QLabel(gpi_matrix_box,tr("Matrix:"),this,"gpi_matrix_label");
  label->setGeometry(20,10,55,21);
  label->setFont(font);
  label->setAlignment(AlignRight|AlignVCenter);
  connect(gpi_matrix_box,SIGNAL(activated(int)),
	  this,SLOT(matrixActivatedData(int)));

  //
  // Input Indicators
  //
  for(int i=0;i<(MAX_GPIO_PINS/8);i++) {
    for(int j=0;j<8;j++) {
      gpi_label[8*i+j]=new QLabel(QString().sprintf("%d",8*i+j+1),this);
      gpi_label[8*i+j]->setGeometry(10+64*j,40+58*i,59,33);
      gpi_label[8*i+j]->setFont(label_font);
      gpi_label[8*i+j]->setAlignment(AlignHCenter|AlignVCenter);
      if(((1>>(8*i+j))&mask)==0) {
	gpi_label[8*i+j]->setPalette(QColor(gray));
      }
      else {
	gpi_label[8*i+j]->setPalette(QColor(green));
      }
      gpi_label[8*i+j]->hide();

      gpi_cart[8*i+j]=new QLabel(this);
      gpi_cart[8*i+j]->setGeometry(10+64*j,73+58*i,59,16);
      gpi_cart[8*i+j]->setFont(font);
      gpi_cart[8*i+j]->setAlignment(AlignHCenter|AlignTop);
      gpi_cart[8*i+j]->hide();
    }
  }

  //
  // Close Button
  //
  gpi_close_button=new QPushButton(this,"gpi_close_button");
  gpi_close_button->setGeometry(sizeHint().width()-90,sizeHint().height()-60,
				80,50);
  gpi_close_button->setFont(font);
  gpi_close_button->setText(tr("&Close"));
  connect(gpi_close_button,SIGNAL(clicked()),this,SLOT(quitMainWidget()));

  //
  // Start Up Timer
  //
  QTimer *timer=new QTimer(this,"start_up_timer");
  connect(timer,SIGNAL(timeout()),this,SLOT(startUpData()));
  timer->start(START_UP_DELAY,true);
}


MainWidget::~MainWidget()
{
}


QSize MainWidget::sizeHint() const
{
  return QSize(528,gpi_grid_height+130);
}


QSizePolicy MainWidget::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
}


void MainWidget::userData()
{
  QString str;

  str=QString(tr("RDGpiMon - User:"));
  setCaption(QString().sprintf("%s %s",(const char *)str,
			       (const char *)gpi_ripc->user()));
}


void MainWidget::matrixActivatedData(int index)
{
  if(gpi_matrix!=NULL) {
    for(int i=0;i<gpi_matrix->gpis();i++) {
      gpi_label[i]->hide();
      gpi_cart[i]->hide();
    }
    delete gpi_matrix;
  }
  gpi_matrix=new RDMatrix(gpi_config->stationName(),index);
  gpi_grid_height=6*gpi_matrix->gpis();
  for(int i=0;i<gpi_matrix->gpis();i++) {
    gpi_label[i]->show();
    gpi_cart[i]->show();
  }
  gpi_grid_height=6*gpi_matrix->gpis();
  setGeometry(geometry().x(),geometry().y(),sizeHint().width(),
	      sizeHint().height());
  gpi_close_button->setGeometry(sizeHint().width()-90,sizeHint().height()-60,
				80,50);
  gpi_ripc->sendGpiStatus(index);
  gpi_ripc->sendGpiMask(index);
  gpi_ripc->sendGpiCart(index);
}


void MainWidget::gpiStateChangedData(int matrix,int line,bool state)
{
  if(matrix!=gpi_matrix->matrix()) {
    return;
  }
  if(state) {
    gpi_label[line]->setPalette(QColor(green));
  }
  else {
    gpi_label[line]->setPalette(QColor(gray));
  }
}


void MainWidget::gpiMaskChangedData(int matrix,int line,bool state)
{
  if(matrix!=gpi_matrix->matrix()) {
    return;
  }
  gpi_label[line]->setEnabled(state);
  gpi_cart[line]->setEnabled(state);
}


void MainWidget::gpiCartChangedData(int matrix,int line,int cartnum)
{
  if(matrix!=gpi_matrix->matrix()) {
    return;
  }
  if(cartnum<=0) {
    gpi_cart[line]->setText("");
  }
  else {
    gpi_cart[line]->setText(QString().sprintf("%06d",cartnum));
  }
}


void MainWidget::startUpData()
{
  matrixActivatedData(0);
}


void MainWidget::quitMainWidget()
{
  exit(0);
}


int main(int argc,char *argv[])
{
  QApplication a(argc,argv);
  
  //
  // Load Translations
  //
  QTranslator qt(0);
  qt.load(QString(QTDIR)+QString("/translations/qt_")+QTextCodec::locale(),
	  ".");
  a.installTranslator(&qt);

  QTranslator libradio(0);
  libradio.load(QString(PREFIX)+QString("/share/srlabs/libradio_")+
		QTextCodec::locale(),".");
  a.installTranslator(&libradio);

  QTranslator tests(0);
  tests.load(QString(PREFIX)+QString("/share/srlabs/rdutils_")+
	     QTextCodec::locale(),".");
  a.installTranslator(&tests);

  //
  // Start Event Loop
  //
  MainWidget *w=new MainWidget(NULL,"main");
  a.setMainWidget(w);
  w->setGeometry(QRect(QPoint(0,0),w->sizeHint()));
  w->show();
  return a.exec();
}
