// rdpanel.cpp
//
// A Dedicated Cart Wall Utility for Rivendell.
//
//   (C) Copyright 2002-2004 Fred Gleason <fredg@paravelsystems.com>
//
//      $Id: rdpanel.cpp,v 1.8 2007/02/14 21:57:04 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License version 2 as
//   published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <stdlib.h>

#include <qmessagebox.h>
#include <qapplication.h>

#include <rdpanel.h>
#include <rd.h>
#include <rdcheck_daemons.h>
#include <rddbheartbeat.h>

#include <globals.h>

//
// Global Resources
//
RDStation *rdstation_conf;
RDAirPlayConf *rdairplay_conf;
RDAudioPort *rdaudioport_conf;
RDUser *rduser;
RDRipc *rdripc;


MainWidget::MainWidget(QWidget *parent,const char *name)
  :QWidget(parent,name)
{
  //
  // Fix the Window Size
  //
#ifndef RESIZABLE
  setMinimumWidth(sizeHint().width());
  setMaximumWidth(sizeHint().width());
  setMinimumHeight(sizeHint().height());
  setMaximumHeight(sizeHint().height());
#endif  // RESIZABLE

  //
  // Generate Fonts
  //
  QFont button_font=QFont("Helvetica",16,QFont::Bold);
  button_font.setPixelSize(16);

  //
  // Ensure that system daemons are running
  //
  RDInitializeDaemons();

  //
  // Load Local Configs
  //
  panel_config=new RDConfig();
  panel_config->load();

  setCaption(QString()
	     .sprintf("RDPanel - Station: %s",(const char *)panel_config->stationName()));

  //
  // Open Database
  //
  panel_db=QSqlDatabase::addDatabase("QMYSQL3");
  if(!panel_db) {
    QMessageBox::warning(this,
	   "Can't Connect","Unable to connect to mySQL Server!",0,1,1);
    exit(0);
  }
  panel_db->setDatabaseName(panel_config->mysqlDbname());
  panel_db->setUserName(panel_config->mysqlUsername());
  panel_db->setPassword(panel_config->mysqlPassword());
  panel_db->setHostName(panel_config->mysqlHostname());
  if(!panel_db->open()) {
    QMessageBox::warning(this,"Can't Connect","Unable to connect to mySQL Server!");
    panel_db->removeDatabase(panel_config->mysqlDbname());
    exit(0);
  }
  new RDDbHeartbeat(panel_config->mysqlHeartbeatInterval(),this);

  //
  // Master Clock Timer
  //
  panel_master_timer=new QTimer(this,"panel_master_timer");
  connect(panel_master_timer,SIGNAL(timeout()),this,SLOT(masterTimerData()));
  panel_master_timer->start(MASTER_TIMER_INTERVAL);

  //
  // CAE Connection
  //
  panel_cae=new RDCae(parent,name);
  panel_cae->connectHost("localhost",CAED_TCP_PORT,panel_config->password());

  //
  // Allocate Global Resources
  //
  rdstation_conf=new RDStation(panel_config->stationName());
  rdairplay_conf=new RDAirPlayConf(panel_config->stationName(),0);

  //
  // Set Audio Assignments
  //
/*
  for(int i=0;i<TOTAL_PORTS;i++) {
    audio_resources->setPort(i,rdairplay_conf->port(i));
    for(int j=0;j<MAX_STREAMS;j++) {
      if(audio_resources->port(i)!=-1) {
	sound_card->setOutputVolume(audio_resources->card(),  
				    j,audio_resources->port(i),-10000);
      }
    }
  }
  RDSetMixerPorts(air_station,sound_card);
*/

  //
  // RIPC Connection
  //
  rdripc=new RDRipc(panel_config->stationName());
  connect(rdripc,SIGNAL(connected(bool)),this,SLOT(connectedData(bool)));
  connect(rdripc,SIGNAL(userChanged()),this,SLOT(userData()));
  connect(rdripc,SIGNAL(rmlReceived(RDMacro *)),
	  this,SLOT(rmlReceivedData(RDMacro *)));
  rdripc->connectHost("localhost",RIPCD_TCP_PORT,panel_config->password());

  //
  // User
  //
  rduser=NULL;

  //
  // Audio Meter
  //
  panel_meter[0]=new RPlayMeter(Radio::Right,this,"left_panel_meter");
  panel_meter[0]->setMode(RSegMeter::Peak);
  panel_meter[0]->setGeometry(30,70,400,20);
  panel_meter[0]->setRange(-3000,-800);
  panel_meter[0]->setSolidBar(-10000);
  panel_meter[0]->setFloatingBar(-10000);
  panel_meter[0]->setHighThreshold(-1600);
  panel_meter[0]->setClipThreshold(-1100);
  panel_meter[0]->setSegmentSize(5);
  panel_meter[0]->setLabel("L");

  panel_meter[1]=new RPlayMeter(Radio::Right,this,"right_panel_meter");
  panel_meter[1]->setMode(RSegMeter::Peak);
  panel_meter[1]->setGeometry(30,90,400,20);
  panel_meter[1]->setRange(-3000,-800);
  panel_meter[1]->setSolidBar(-10000);
  panel_meter[1]->setFloatingBar(-10000);
  panel_meter[1]->setHighThreshold(-1600);
  panel_meter[1]->setClipThreshold(-1100);
  panel_meter[1]->setSegmentSize(5);
  panel_meter[1]->setLabel("R");

  //
  // Meter Timer
  //
  QTimer *timer=new QTimer(this,"meter_timer");
  connect(timer,SIGNAL(timeout()),this,SLOT(meterData()));
  timer->start(METER_INTERVAL);

  //
  // Sound Panel Array
  //
/*
  panel_panel=new SoundPanel(panel_cae,this,"panel_panel");
  panel_panel->setGeometry(510,120,panel_panel->sizeHint().width(),
			 panel_panel->sizeHint().height());
  connect(rdripc,SIGNAL(userChanged()),panel_panel,SLOT(changeUser()));
  connect(panel_master_timer,SIGNAL(timeout()),panel_panel,SLOT(tickClock()));
*/
}


QSize MainWidget::sizeHint() const
{
  return QSize(1024,738);
}


void MainWidget::rmlReceivedData(RDMacro *rml)
{
  RunLocalMacros(rml);
}


QSizePolicy MainWidget::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
}


void MainWidget::userData()
{
  if(rduser!=NULL) {
    delete rduser;
  }
  rduser=new RDUser(rdripc->user());
  SetCaption();
}


void MainWidget::meterData()
{
#ifdef SHOW_METER_SLOTS
  printf("meterData()\n");
#endif
  double ratio[2]={0.0,0.0};
  short level[2];
/*
  for(int i=0;i<AIR_TOTAL_PORTS;i++) {
    sound_card->
      outputPortMeter(audio_resources->card(),audio_resources->port(i),level);
    for(int j=0;j<2;j++) {
      ratio[j]+=pow(10.0,((double)level[j])/1000.0);
    }
  }
  for(int i=0;i<2;i++) {
    panel_meter[i]->setPeakBar((int)(log10(ratio[i])*1000.0));
  }
*/
}


void MainWidget::masterTimerData()
{
}


void MainWidget::closeEvent(QCloseEvent *e)
{
  panel_db->removeDatabase(panel_config->mysqlDbname());
  exit(0);
}


void MainWidget::RunLocalMacros(RDMacro *rml)
{
}


void MainWidget::SetCaption()
{
  setCaption(QString().sprintf("RDAirPlay - Station: %s  User: %s",
			       (const char *)panel_config->stationName(),
			       (const char *)rdripc->user()));
}


int main(int argc,char *argv[])
{
  QApplication a(argc,argv);
  
  MainWidget *w=new MainWidget(NULL,"main");
  a.setMainWidget(w);
  w->setGeometry(QRect(QPoint(0,0),w->sizeHint()));
  w->show();
  return a.exec();
}
