// rdcut_dialog.cpp
//
// A widget to select a Rivendell Cut.
//
//   (C) Copyright 2002-2004 Fred Gleason <fredg@paravelsystems.com>
//
//      $Id: rdcut_dialog.cpp,v 1.23 2007/07/09 07:21:12 stgabriel Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License version 2 as
//   published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <qpushbutton.h>
#include <qlabel.h>
#include <qsqlquery.h>

#include <rdcut_dialog.h>
#include <rdcart_search_text.h>
#include <rdtextvalidator.h>
#include <rdadd_cart.h>

//
// Icons
//
#include "../icons/play.xpm"
#include "../icons/rml5.xpm"


RDCutDialog::RDCutDialog(QString *cutname,QString *filter,QString *group,
			 QString username,bool show_clear,bool allow_add,
			 bool exclude_tracks,QWidget *parent,const char *name)
  : QDialog(parent,name,true)
{
  cut_cutname=cutname;
  cut_exclude_tracks=exclude_tracks;
  cut_group=group;
  cut_username=username;
  cut_allow_clear=show_clear;

  if(filter==NULL) {
    cut_filter=new QString();
    local_filter=true;
  }
  else {
    cut_filter=filter;
    local_filter=false;
  }

  setCaption(tr("Select Cut"));

  //
  // Create Icons
  //
  cut_playout_map=new QPixmap(play_xpm);
  cut_macro_map=new QPixmap(rml5_xpm);

  //
  // Generate Fonts
  //
  QFont label_font("Helvetica",12,QFont::Bold);
  label_font.setPixelSize(12);

  //
  // Fix the Window Size
  //
  setMinimumWidth(sizeHint().width());
  setMaximumWidth(sizeHint().width());
  setMinimumHeight(sizeHint().height());
  setMaximumHeight(sizeHint().height());

  //
  // Text Validator
  //
  RDTextValidator *validator=new RDTextValidator(this,"validator");

  //
  // Filter Selector
  //
  cut_filter_edit=new QLineEdit(this,"cut_filter_edit");
  cut_filter_edit->setGeometry(100,10,360,20);
  cut_filter_edit->setValidator(validator);
  QLabel *label=new QLabel(cut_filter_edit,tr("Cart Filter:"),
			   this,"cut_filter_label");
  label->setGeometry(10,10,85,20);
  label->setAlignment(AlignRight|AlignVCenter);
  label->setFont(label_font);
  connect(cut_filter_edit,SIGNAL(textChanged(const QString &)),
	  this,SLOT(filterChangedData(const QString &)));

  //
  // Filter Clear Button
  //
  cut_clear_button=new QPushButton(this,"cut_clear_button");
  cut_clear_button->setGeometry(470,8,60,24);
  cut_clear_button->setText(tr("C&lear"));
  connect(cut_clear_button,SIGNAL(clicked()),this,SLOT(clearData()));

  //
  // Group Selector
  //
  cut_group_box=new QComboBox(this,"cut_clear_box");
  cut_group_box->setGeometry(100,35,160,20);
  label=new QLabel(cut_filter_edit,tr("Group:"),this,"cut_group_label");
  label->setGeometry(10,35,85,20);
  label->setAlignment(AlignRight|AlignVCenter);
  label->setFont(label_font);
  connect(cut_group_box,SIGNAL(activated(const QString &)),
	  this,SLOT(groupActivatedData(const QString &)));

  //
  // Cart List
  //
  cut_cart_list=new RDListView(this,"cut_cart_list");
  cut_cart_list->setGeometry(10,90,300,200);
  cut_cart_list->setAllColumnsShowFocus(true);
  cut_cart_list->setItemMargin(5);
  connect(cut_cart_list,SIGNAL(selectionChanged()),
	  this,SLOT(selectionChangedData()));
  label=new QLabel(cut_cart_list,tr("Carts"),this,"cut_cart_label");
  label->setGeometry(15,70,100,20);
  label->setFont(label_font);
  cut_cart_list->addColumn("");
  cut_cart_list->setColumnAlignment(0,Qt::AlignHCenter);
  cut_cart_list->addColumn(tr("NUMBER"));
  cut_cart_list->setColumnAlignment(1,Qt::AlignHCenter);
  cut_cart_list->addColumn(tr("TITLE"));
  cut_cart_list->setColumnAlignment(2,Qt::AlignLeft);
  cut_cart_list->addColumn(tr("GROUP"));
  cut_cart_list->setColumnAlignment(3,Qt::AlignCenter);

  //
  // Cut List
  //
  cut_cut_list=new QListView(this,"cut_cut_list");
  cut_cut_list->setGeometry(320,90,sizeHint().width()-330,200);
  cut_cut_list->setAllColumnsShowFocus(true);
  cut_cut_list->setItemMargin(5);
  label=new QLabel(cut_cut_list,tr("Cuts"),this,"cut_cut_label");
  label->setGeometry(325,70,100,20);
  label->setFont(label_font);
  cut_cut_list->addColumn(tr("DESCRIPTION"));
  cut_cut_list->setColumnAlignment(0,Qt::AlignLeft);
  cut_cut_list->addColumn(tr("NUMBER"));
  cut_cut_list->setColumnAlignment(1,Qt::AlignLeft);


  QPushButton *button=NULL;

  //
  // Add Button
  //
  button=new QPushButton(tr("&Add New\nCart"),this,"add_button");
  button->setGeometry(10,sizeHint().height()-60,80,50);
  button->setFont(label_font);
  connect(button,SIGNAL(clicked()),this,SLOT(addButtonData()));
  if(!allow_add) {
    button->hide();
  }

  //
  // Clear Button
  //
  button=new QPushButton(tr("&Clear"),this,"clear_button");
  button->setFont(label_font);
  connect(button,SIGNAL(clicked()),this,SLOT(clearButtonData()));
  if(!show_clear) {
    button->hide();
  }
  if(allow_add) {
    button->setGeometry(100,sizeHint().height()-60,80,50);
  }
  else {
    button->setGeometry(10,sizeHint().height()-60,80,50);
  }

  //
  // OK Button
  //
  button=new QPushButton(tr("&OK"),this,"ok_button");
  button->setGeometry(sizeHint().width()-180,sizeHint().height()-60,80,50);
  button->setFont(label_font);
  button->setDefault(true);
  connect(button,SIGNAL(clicked()),this,SLOT(okData()));

  //
  // Cancel Button
  //
  button=new QPushButton(tr("&Cancel"),this,"cancel_button");
  button->setGeometry(sizeHint().width()-90,sizeHint().height()-60,80,50);
  button->setFont(label_font);
  connect(button,SIGNAL(clicked()),this,SLOT(cancelData()));

  //
  // Populate Data
  //
  BuildGroupList();
  cut_filter_edit->setText(*cut_filter);
  RefreshCarts();
  RefreshCuts();
  SelectCut(*cut_cutname);
}


RDCutDialog::~RDCutDialog()
{
  if(local_filter) {
    delete cut_filter;
  }
}


QSize RDCutDialog::sizeHint() const
{
  return QSize(550,370);
}


QSizePolicy RDCutDialog::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
}


void RDCutDialog::filterChangedData(const QString &str)
{
  if(str.isEmpty()) {
    cut_clear_button->setDisabled(true);
  }
  else {
    cut_clear_button->setEnabled(true);
  }
  RefreshCarts();
  RefreshCuts();
}


void RDCutDialog::clearData()
{
  cut_filter_edit->clear();
  filterChangedData("");
}


void RDCutDialog::groupActivatedData(const QString &str)
{
  RefreshCarts();
  RefreshCuts();
}


void RDCutDialog::selectionChangedData()
{
  RefreshCuts();
  if(cut_cut_list->firstChild()!=NULL) {
    cut_cut_list->setSelected(cut_cut_list->firstChild(),true);
  }
}


void RDCutDialog::clearButtonData()
{
  RDListViewItem *item=(RDListViewItem *)cut_cart_list->selectedItem();
  if(item!=NULL) {
    cut_cart_list->setSelected(item,false);
  }
  cut_cut_list->clear();
}


void RDCutDialog::addButtonData()
{
  QString cart_group=cut_group_box->currentText();
  RDCart::Type cart_type=RDCart::Audio;
  QString cart_title;
  QString sql;
  QSqlQuery *q;
  int cart_num=-1;

  RDAddCart *add_cart=new RDAddCart(&cart_group,&cart_type,&cart_title,
				    cut_username,this);
  if((cart_num=add_cart->exec())<0) {
    delete add_cart;
    return;
  }
  sql=QString().sprintf("insert into CART set \
                         NUMBER=%d,TYPE=%d,GROUP_NAME=\"%s\",TITLE=\"%s\"",
			cart_num,cart_type,
			(const char *)cart_group,
			(const char *)cart_title);
  q=new QSqlQuery(sql);
  delete q;
  sql=QString().sprintf("insert into CUTS set CUT_NAME=\"%06d_001\",\
                         CART_NUMBER=%d,DESCRIPTION=\"Cut 001\"",
			cart_num,cart_num);
  q=new QSqlQuery(sql);
  delete q;

  RDListViewItem *item=new RDListViewItem(cut_cart_list);
  item->setPixmap(0,*cut_playout_map);
  item->setText(1,QString().sprintf("%06d",cart_num));
  item->setText(2,cart_title);
  cut_cart_list->setSelected(item,true);
  cut_cart_list->ensureItemVisible(item);

  delete add_cart;
}


void RDCutDialog::okData()
{
  RDListViewItem *cart_item=(RDListViewItem *)cut_cart_list->selectedItem();
  QListViewItem *cut_item=cut_cut_list->selectedItem();
  if((cart_item==NULL)||(cut_item==NULL)) {
    if(cut_allow_clear) {
      *cut_cutname="";
      if(!local_filter) {
	*cut_filter=cut_filter_edit->text();
      }
      if(cut_group!=NULL) {
	*cut_group=cut_group_box->currentText();
      }
      done(0);
    }
    return;
  }

  *cut_cutname=cart_item->text(1)+QString("_")+cut_item->text(1);
  if(!local_filter) {
    *cut_filter=cut_filter_edit->text();
  }
  if(cut_group!=NULL) {
    *cut_group=cut_group_box->currentText();
  }
  done(0);
}


void RDCutDialog::cancelData()
{
  done(1);
}


void RDCutDialog::closeEvent(QCloseEvent *e)
{
  cancelData();
}


void RDCutDialog::RefreshCarts()
{
  QString sql;
  QSqlQuery *q;
  RDListViewItem *l;
  QString group=cut_group_box->currentText();

  if(!cut_cutname->isEmpty()) {
  }
  cut_cart_list->clear();
  if(group==QString("ALL")) {
    group="";
  }
  sql=QString().sprintf("select CART.NUMBER,CART.TITLE,CART.GROUP_NAME,\
                         GROUPS.COLOR from CART left join GROUPS \
                         on CART.GROUP_NAME=GROUPS.NAME \
                         where (%s)&&(CART.TYPE=1)",
	      (const char *)RDCartSearchText(cut_filter_edit->text(),group).utf8());
  if(cut_exclude_tracks) {
    sql+="&&(CART.OWNER is null)";
  }
  q=new QSqlQuery(sql);
  while(q->next()) {
    l=new RDListViewItem(cut_cart_list);
    l->setPixmap(0,*cut_playout_map);
    l->setText(1,QString().sprintf("%06u",q->value(0).toUInt()));   // Number

    l->setText(2,q->value(1).toString());     // Title
    l->setText(3,q->value(2).toString());     // Group
    l->setTextColor(3,q->value(3).toString(),QFont::Bold);
  }
  delete q;
}



void RDCutDialog::RefreshCuts()

{
  QString sql;
  QSqlQuery *q;
  QListViewItem *l;
  QListViewItem *cart_item=cut_cart_list->selectedItem();

  if(cart_item==NULL) {
    return;
  }
  cut_cut_list->clear();
  sql=QString().sprintf("select DESCRIPTION,CUT_NAME from CUTS where \
CART_NUMBER=%s",(const char *)cart_item->text(1));
  q=new QSqlQuery(sql);
  while(q->next()) {
    l=new QListViewItem(cut_cut_list);
    l->setText(0,q->value(0).toString());     // Description
    l->setText(1,QString().sprintf("%03u",    // Cut Number
				   q->value(1).toString().right(3).toUInt()));
  }
  delete q;
}


void RDCutDialog::SelectCut(QString cutname)
{
  QString cart=cutname.left(6);
  QString cut=cutname.right(3);
  QListViewItem *item=cut_cart_list->findItem(cart,1);
  if(item!=NULL) {
    cut_cart_list->setSelected(item,true);
    cut_cart_list->ensureItemVisible(item);
  }
  RefreshCuts();
  item=cut_cut_list->findItem(cut,1);
  if(item!=NULL) {
    cut_cut_list->setSelected(item,true);
  }
}


void RDCutDialog::BuildGroupList()
{
  QString sql;
  QSqlQuery *q;
  
  cut_group_box->clear();
  cut_group_box->insertItem(tr("ALL"));
  if(cut_username.isEmpty()) {
    sql="select NAME from GROUPS order by NAME desc";
  }
  else {
    sql=QString().sprintf("select GROUP_NAME from USER_PERMS\
                           where USER_NAME=\"%s\" order by GROUP_NAME",
			  (const char *)cut_username);
  }
  q=new QSqlQuery(sql);
  while(q->next()) {
    cut_group_box->insertItem(q->value(0).toString(),true);
  }
  delete q;

  //
  // Preselect Group
  //
  if(cut_group!=NULL) {
    for(int i=0;i<cut_group_box->count();i++) {
      if(*cut_group==cut_group_box->text(i)) {
	cut_group_box->setCurrentItem(i);
	return;
      }
    }
  }
}
