// list_gpis.cpp
//
// List a Rivendell GPIs
//
//   (C) Copyright 2002-2004 Fred Gleason <fredg@paravelsystems.com>
//
//      $Id: list_gpis.cpp,v 1.7 2007/09/14 14:06:27 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License version 2 as
//   published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <qdialog.h>
#include <qstring.h>
#include <qtextedit.h>
#include <qpainter.h>
#include <qmessagebox.h>
#include <qsqldatabase.h>

#include <rd.h>
#include <rdpasswd.h>
#include <rdmacro.h>
#include <rdstation.h>

#include <list_gpis.h>
#include <edit_gpi.h>
#include <globals.h>


ListGpis::ListGpis(RDMatrix *matrix,QWidget *parent,const char *name)
  : QDialog(parent,name,true)
{
  QString sql;
  QSqlQuery *q;
  QListViewItem *l;

  list_matrix=matrix;
  list_size=list_matrix->gpis();
  setCaption(tr("List GPIs"));

  //
  // Fix the Window Size
  //
  setMinimumWidth(sizeHint().width());
  setMaximumWidth(sizeHint().width());
  setMinimumHeight(sizeHint().height());
  setMaximumHeight(sizeHint().height());

  //
  // Create Fonts
  //
  QFont bold_font=QFont("Helvetica",12,QFont::Bold);
  bold_font.setPixelSize(12);
  QFont font=QFont("Helvetica",12,QFont::Normal);
  font.setPixelSize(12);

  //
  // Gpis List Box
  //
  list_list_view=new QListView(this,"list_box");
  list_list_view->
    setGeometry(10,24,sizeHint().width()-20,sizeHint().height()-94);
  QLabel *label=
    new QLabel(list_list_view,tr("GPI Lines"),this,"list_list_view_label");
  label->setFont(bold_font);
  label->setGeometry(14,5,85,19);
  list_list_view->setAllColumnsShowFocus(true);
  list_list_view->setItemMargin(5);
  connect(list_list_view,
	  SIGNAL(doubleClicked(QListViewItem *,const QPoint &,int)),
	  this,
	  SLOT(doubleClickedData(QListViewItem *,const QPoint &,int)));

  list_list_view->addColumn(tr("GPI"));
  list_list_view->setColumnAlignment(0,Qt::AlignHCenter);

  list_list_view->addColumn(tr("MACRO CART"));
  list_list_view->setColumnAlignment(1,Qt::AlignHCenter);

  list_list_view->addColumn(tr("DESCRIPTION"));
  list_list_view->setColumnAlignment(2,Qt::AlignLeft);

  //
  //  Edit Button
  //
  QPushButton *button=new QPushButton(this,"edit_button");
  button->setGeometry(10,sizeHint().height()-60,80,50);
  button->setFont(bold_font);
  button->setText(tr("&Edit"));
  connect(button,SIGNAL(clicked()),this,SLOT(editData()));

  //
  //  Ok Button
  //
  button=new QPushButton(this,"ok_button");
  button->setGeometry(sizeHint().width()-180,sizeHint().height()-60,80,50);
  button->setDefault(true);
  button->setFont(bold_font);
  button->setText(tr("&OK"));
  connect(button,SIGNAL(clicked()),this,SLOT(okData()));

  //
  //  Cancel Button
  //
  button=new QPushButton(this,"cancel_button");
  button->setGeometry(sizeHint().width()-90,sizeHint().height()-60,
			     80,50);
  button->setFont(bold_font);
  button->setText(tr("&Cancel"));
  connect(button,SIGNAL(clicked()),this,SLOT(cancelData()));

  //
  // Load Values
  //
  sql=QString().sprintf("select GPIS.NUMBER,GPIS.MACRO_CART,CART.TITLE \
                         from GPIS,CART where (GPIS.STATION_NAME=\"%s\") && \
                         (GPIS.MATRIX=%d)&&(GPIS.MACRO_CART=CART.NUMBER)\
                         order by NUMBER",
			(const char *)list_matrix->station(),
			list_matrix->matrix());
  
  q=new QSqlQuery(sql);
  q->first();
  for(int i=0;i<list_size;i++) {
    l=new QListViewItem(list_list_view); 
    l->setText(0,QString().sprintf("%03d",i+1));
    if(q->isValid()&&(q->value(0).toInt()==(i+1))){
      if(q->value(1).toInt()>0) {
	l->setText(1,QString().sprintf("%06d",q->value(1).toInt()));
      }
      if(q->value(1).toInt()>0) {
	l->setText(2,q->value(2).toString());
      }
      else {
	l->setText(2,tr("[unassigned]"));
      }
      q->next();
    }
    else {
      l->setText(2,tr("[unassigned]"));
    }
  }
  delete q;
}


QSize ListGpis::sizeHint() const
{
  return QSize(400,250);
} 


QSizePolicy ListGpis::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
}


void ListGpis::editData()
{
  int cart=-1;

  QString desc;
  QListViewItem *item=list_list_view->selectedItem();
  if(item==NULL) {
    return;
  }
  int gpi=item->text(0).toInt();
  if(!item->text(1).isEmpty()) {
    cart=item->text(1).toInt();
  }
  EditGpi *gpi_box=new EditGpi(gpi,&cart,&desc,this,"gpi_box");
  if(gpi_box->exec()==0) {
    if(cart>0) {
      item->setText(1,QString().sprintf("%06d",cart));
      item->setText(2,desc);
    }
    else {
      item->setText(1,"");
      item->setText(2,tr("[unassigned]"));
    }
  }
  delete gpi_box;
}


void ListGpis::doubleClickedData(QListViewItem *,const QPoint &,int)
{
  editData();
}


void ListGpis::okData()
{
  QString sql;
  QSqlQuery *q;
  RDMacro rml;

  sql=QString().sprintf("delete from GPIS where (STATION_NAME=\"%s\")&&\
                           MATRIX=%d",
			(const char *)list_matrix->station(),
			list_matrix->matrix());
  q=new QSqlQuery(sql);
  delete q;
  RDStation *station=new RDStation(list_matrix->station());
  rml.setCommand(RDMacro::GI);
  rml.setRole(RDMacro::Cmd);
  rml.setArgQuantity(4);
  rml.setArg(0,list_matrix->matrix());
  rml.setAddress(station->address());
  rml.setEchoRequested(false);
  delete station;

  QListViewItem *item=list_list_view->firstChild();
  while(item!=NULL) {
    sql=QString().sprintf("insert into GPIS set STATION_NAME=\"%s\",\
                           MATRIX=%d,NUMBER=%d,MACRO_CART=%d",
			  (const char *)list_matrix->station(),
			  list_matrix->matrix(),
			  item->text(0).toInt(),item->text(1).toInt());
    q=new QSqlQuery(sql);
    delete q;
    rml.setArg(1,item->text(0).toInt());
    rml.setArg(2,true);
    rml.setArg(3,item->text(1).toInt());
    rdripc->sendRml(&rml);
    item=item->nextSibling();
  }
  done(0);
}


void ListGpis::cancelData()
{
  done(1);
}
