// rdrecording.cpp
//
// Abstract a Rivendell Netcatch Recording.
//
//   (C) Copyright 2002-2004 Fred Gleason <fredg@paravelsystems.com>
//
//      $Id: rdrecording.cpp,v 1.21 2007/09/14 14:06:24 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License version 2 as
//   published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <unistd.h>

#include <rd.h>
#include <rdconf.h>

#include <rdrecording.h>
#include <rddeck.h>

//
// Global Classes
//
RDRecording::RDRecording(int id,bool create)
{
  QSqlQuery *q;
  QString sql;

  if(id<0) {
    rec_id=AddRecord();
    create=false;
  }
  else {
    rec_id=id;
  }

  if(create) {
    sql=QString().sprintf("select ID from RECORDINGS where ID=%d",rec_id);
    q=new QSqlQuery(sql);
    if(q->size()!=1) {
      delete q;
      sql=QString().sprintf("insert into RECORDINGS set ID=%d",rec_id);
      q=new QSqlQuery(sql);
    }
    delete q;
  }
}


int RDRecording::id() const
{
  return rec_id;
}


bool RDRecording::isActive()
{
  return RDBool(GetStringValue("IS_ACTIVE"));
}


void RDRecording::setIsActive(bool state)
{
  SetRow("IS_ACTIVE",RDYesNo(state));
}


QString RDRecording::station()
{
  return GetStringValue("STATION_NAME");
}


void RDRecording::setStation(QString name)
{
  SetRow("STATION_NAME",name);
}


RDRecording::Type RDRecording::type()
{
  return (RDRecording::Type)GetIntValue("TYPE");
}


void RDRecording::setType(RDRecording::Type type)
{
  SetRow("TYPE",(int)type);
}


unsigned RDRecording::channel()
{
  return GetUIntValue("CHANNEL");
}


void RDRecording::setChannel(unsigned chan)
{
  SetRow("CHANNEL",chan);
}


QTime RDRecording::startTime()
{
  return GetTimeValue("START_TIME");
}


void RDRecording::setStartTime(QTime time)
{
  SetRow("START_TIME",time);
}


QTime RDRecording::endTime()
{
  return GetTimeValue("END_TIME");
}


void RDRecording::setEndTime(QTime time)
{
  SetRow("END_TIME",time);
}


QString RDRecording::cutName()
{
  return GetStringValue("CUT_NAME");
}


void RDRecording::setCutName(QString name)
{
  SetRow("CUT_NAME",name);
}


bool RDRecording::sun()
{
  return GetBoolValue("SUN");
}


void RDRecording::setSun(bool state)
{
  SetRow("SUN",state);
}


bool RDRecording::mon()
{
  return GetBoolValue("MON");
}


void RDRecording::setMon(bool state)
{
  SetRow("MON",state);
}


bool RDRecording::tue()
{
  return GetBoolValue("TUE");
}


void RDRecording::setTue(bool state)
{
  SetRow("TUE",state);
}


bool RDRecording::wed()
{
  return GetBoolValue("WED");
}


void RDRecording::setWed(bool state)
{
  SetRow("WED",state);
}


bool RDRecording::thu()
{
  return GetBoolValue("THU");
}


void RDRecording::setThu(bool state)
{
  SetRow("THU",state);
}


bool RDRecording::fri()
{
  return GetBoolValue("FRI");
}


void RDRecording::setFri(bool state)
{
  SetRow("FRI",state);
}


bool RDRecording::sat()
{
  return GetBoolValue("SAT");
}


void RDRecording::setSat(bool state)
{
  SetRow("SAT",state);
}


int RDRecording::switchSource()
{
  return GetIntValue("SWITCH_INPUT");
}


void RDRecording::setSwitchSource(int input)
{
  SetRow("SWITCH_INPUT",input);
}


int RDRecording::switchDestination()
{
  return GetIntValue("SWITCH_OUTPUT");
}


void RDRecording::setSwitchDestination(int output)
{
  SetRow("SWITCH_OUTPUT",output);
}


QString RDRecording::description()
{
  return GetStringValue("DESCRIPTION");
}


void RDRecording::setDescription(QString string)
{
  SetRow("DESCRIPTION",string);
}


unsigned RDRecording::length()
{
  return GetUIntValue("LENGTH");
}


void RDRecording::setLength(unsigned length)
{
  SetRow("LENGTH",length);
}


int RDRecording::startGpi()
{
  return GetIntValue("START_GPI");
}


void RDRecording::setStartGpi(int line)
{
  SetRow("START_GPI",line);
}


int RDRecording::endGpi()
{
  return GetIntValue("END_GPI");
}


void RDRecording::setEndGpi(int line)
{
  SetRow("END_GPI",line);
}


bool RDRecording::allowMultipleRecordings()
{
  return RDBool(GetStringValue("ALLOW_MULT_RECS"));
}


void RDRecording::setAllowMultipleRecordings(bool state)
{
  SetRow("ALLOW_MULT_RECS",state);
}


unsigned RDRecording::maxGpiRecordingLength()
{
  return GetUIntValue("MAX_GPI_REC_LENGTH");
}


void RDRecording::setMaxGpiRecordingLength(unsigned len)
{
  SetRow("MAX_GPI_REC_LENGTH",len);
}


unsigned RDRecording::trimThreshold()
{
  return GetUIntValue("TRIM_THRESHOLD");
}


void RDRecording::setTrimThreshold(unsigned level)
{
  SetRow("TRIM_THRESHOLD",level);
}


unsigned RDRecording::startdateOffset()
{
  return GetUIntValue("STARTDATE_OFFSET");
}


void RDRecording::setStartdateOffset(unsigned offset)
{
  SetRow("STARTDATE_OFFSET",offset);
}


unsigned RDRecording::enddateOffset()
{
  return GetUIntValue("ENDDATE_OFFSET");
}


void RDRecording::setEnddateOffset(unsigned offset)
{
  SetRow("ENDDATE_OFFSET",offset);
}


RDSettings::Format RDRecording::format()
{
  return (RDSettings::Format)GetIntValue("FORMAT");
}


void RDRecording::setFormat(RDSettings::Format fmt)
{
  SetRow("FORMAT",(int)fmt);
}


int RDRecording::channels()
{
  return GetIntValue("CHANNELS");
}


void RDRecording::setChannels(int chan)
{
  SetRow("CHANNELS",chan);
}


int RDRecording::samprate()
{
  return GetIntValue("SAMPRATE");
}


void RDRecording::setSamprate(int rate)
{
  SetRow("SAMPRATE",rate);
}


int RDRecording::bitrate()
{
  return GetIntValue("BITRATE");
}


int RDRecording::quality()
{
  return GetIntValue("QUALITY");
}


void RDRecording::setQuality(int qual)
{
  SetRow("QUALITY",qual);
}


void RDRecording::setBitrate(int rate)
{
  SetRow("BITRATE",rate);
}


int RDRecording::normalizationLevel()
{
  return GetIntValue("NORMALIZE_LEVEL");
}


void RDRecording::setNormalizationLevel(int level)
{
  SetRow("NORMALIZE_LEVEL",level);
}


int RDRecording::macroCart()
{
  return GetIntValue("MACRO_CART");
}


void RDRecording::setMacroCart(int cart)
{
  SetRow("MACRO_CART",cart);
}


bool RDRecording::oneShot()
{
  return GetBoolValue("ONE_SHOT");
}


void RDRecording::setOneShot(bool state)
{
  SetRow("ONE_SHOT",state);
}


RDRecording::StartType RDRecording::startType()
{
  return (RDRecording::StartType)GetUIntValue("START_TYPE");
}


void RDRecording::setStartType(StartType type)
{
  SetRow("START_TYPE",(unsigned)type);
}


RDRecording::EndType RDRecording::endType()
{
  return (RDRecording::EndType)GetUIntValue("END_TYPE");
}


void RDRecording::setEndType(EndType type)
{
  SetRow("END_TYPE",(unsigned)type);
}


int RDRecording::startMatrix()
{
  return GetIntValue("START_MATRIX");
}


void RDRecording::setStartMatrix(int matrix)
{
  SetRow("START_MATRIX",matrix);
}


int RDRecording::startLine()
{
  return GetIntValue("START_LINE");
}


void RDRecording::setStartLine(int line)
{
  SetRow("START_LINE",line);
}


int RDRecording::endMatrix()
{
  return GetIntValue("END_MATRIX");
}


void RDRecording::setEndMatrix(int matrix)
{
  SetRow("END_MATRIX",matrix);
}


int RDRecording::endLine()
{
  return GetIntValue("END_LINE");
}


void RDRecording::setEndLine(int line)
{
  SetRow("END_LINE",line);
}


int RDRecording::startLength()
{
  return GetIntValue("START_LENGTH");
}


void RDRecording::setStartLength(int len)
{
  SetRow("START_LENGTH",len);
}


int RDRecording::endLength()
{
  return GetIntValue("END_LENGTH");
}


void RDRecording::setEndLength(int len)
{
  SetRow("END_LENGTH",len);
}


int RDRecording::startOffset()
{
  return GetIntValue("START_OFFSET");
}


void RDRecording::setStartOffset(int offset)
{
  SetRow("START_OFFSET",offset);
}


QString RDRecording::url()
{
  return GetStringValue("URL");
}


void RDRecording::setUrl(QString url)
{
  SetRow("URL",url);
}


QString RDRecording::urlUsername()
{
  return GetStringValue("URL_USERNAME");
}


void RDRecording::setUrlUsername(QString name)
{
  SetRow("URL_USERNAME",name);
}


QString RDRecording::urlPassword()
{
  return GetStringValue("URL_PASSWORD");
}


void RDRecording::setUrlPassword(QString passwd)
{
  SetRow("URL_PASSWORD",passwd);
}


QString RDRecording::typeString(RDRecording::Type type)
{
  QString str;

  switch(type) {
      case RDRecording::Recording:
	str=QT_TR_NOOP("Recording");
	break;

      case RDRecording::MacroEvent:
	str=QT_TR_NOOP("Macro Event");
	break;

      case RDRecording::SwitchEvent:
	str=QT_TR_NOOP("Switch Event");
	break;

      case RDRecording::Playout:
	str=QT_TR_NOOP("Playout");
	break;

      case RDRecording::Download:
	str=QT_TR_NOOP("Download");
	break;

      case RDRecording::Upload:
	str=QT_TR_NOOP("Upload");
	break;
  }

  return str;
}


int RDRecording::GetIntValue(QString field)
{
  QString sql;
  QSqlQuery *q;
  int accum;
  
  sql=QString().sprintf("select %s from RECORDINGS where ID=%d",
			(const char *)field,rec_id);
  q=new QSqlQuery(sql);
  if(q->first()) {
    accum=q->value(0).toInt();
    delete q;
    return accum;
  }
  delete q;
  return 0;    
}


unsigned RDRecording::GetUIntValue(QString field)
{
  QString sql;
  QSqlQuery *q;
  int accum;
  
  sql=QString().sprintf("select %s from RECORDINGS where ID=%d",
			(const char *)field,rec_id);
  q=new QSqlQuery(sql);
  if(q->first()) {
    accum=q->value(0).toUInt();
    delete q;
    return accum;
  }
  delete q;
  return 0;    
}


bool RDRecording::GetBoolValue(QString field)
{
  QString sql;
  QSqlQuery *q;
  bool state;

  sql=QString().sprintf("select %s from RECORDINGS where ID=%d",
			(const char *)field,rec_id);
  q=new QSqlQuery(sql);
  if(q->first()) {
    state=RDBool(q->value(0).toString());
    delete q;
    return state;
  }
  delete q;
  return false;    
}


QString RDRecording::GetStringValue(QString field)
{
  QString sql;
  QSqlQuery *q;
  QString accum;

  sql=QString().sprintf("select %s from RECORDINGS where ID=%d",
			(const char *)field,rec_id);
  q=new QSqlQuery(sql);
  if(q->first()) {
    accum=q->value(0).toString();
    delete q;
    return accum; 
  }
  delete q;
  return QString();    
}


QTime RDRecording::GetTimeValue(QString field)
{
  QString sql;
  QSqlQuery *q;
  QTime accum;

  sql=QString().sprintf("select %s from RECORDINGS where ID=%d",
			(const char *)field,rec_id);
  q=new QSqlQuery(sql);
  if(q->first()) {
    accum=q->value(0).toTime();
    delete q;
    return accum; 
  }
  delete q;
  return QTime();    
}


int RDRecording::AddRecord()
{
  QString sql;
  QSqlQuery *q;
  int n;

  sql=QString("select ID from RECORDINGS order by ID desc limit 1");
  q=new QSqlQuery(sql);
  if(q->first()) {
    n=q->value(0).toInt()+1;
  }
  else {
    n=1;
  }
  delete q;
  sql=QString().sprintf("insert into RECORDINGS set ID=%d",n);
  q=new QSqlQuery(sql);
  delete q;
  return n;
}


void RDRecording::SetRow(QString param,int value)
{
  QSqlQuery *q;
  QString sql;

  sql=QString().sprintf("update RECORDINGS set %s=%d where ID=%d",
			(const char *)param,value,rec_id);
  q=new QSqlQuery(sql);
  delete q;
}


void RDRecording::SetRow(QString param,unsigned value)
{
  QSqlQuery *q;
  QString sql;

  sql=QString().sprintf("update RECORDINGS set %s=%u where ID=%d",
			(const char *)param,value,rec_id);
  q=new QSqlQuery(sql);
  delete q;
}


void RDRecording::SetRow(QString param,bool value)
{
  QSqlQuery *q;
  QString sql;

  sql=QString().sprintf("update RECORDINGS set %s=\'%s\' where ID=%d",
			(const char *)param,(const char *)RDYesNo(value),
			rec_id);
  q=new QSqlQuery(sql);
  delete q;
}


void RDRecording::SetRow(QString param,QString value)
{
  QSqlQuery *q;
  QString sql;

  sql=QString().sprintf("update RECORDINGS set %s=\"%s\" where ID=%d",
			(const char *)param,(const char *)value,rec_id);
  q=new QSqlQuery(sql);
  delete q;
}


void RDRecording::SetRow(QString param,QTime value)
{
  QSqlQuery *q;
  QString sql;

  sql=QString().sprintf("update RECORDINGS set %s=\"%s\" where ID=%d",
			(const char *)param,
			(const char *)value.toString("hh:mm:ss"),rec_id);
  q=new QSqlQuery(sql);
  delete q;
}
