// list_gpis.cpp
//
// List a Rivendell GPIOs
//
//   (C) Copyright 2002-2008 Fred Gleason <fredg@paravelsystems.com>
//
//      $Id: list_gpis.cpp,v 1.10 2008-09-30 18:19:41 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License version 2 as
//   published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <qdialog.h>
#include <qstring.h>
#include <qtextedit.h>
#include <qpainter.h>
#include <qmessagebox.h>

#include <rd.h>
#include <rdpasswd.h>
#include <rdmacro.h>
#include <rdstation.h>
#include <rddb.h>

#include <list_gpis.h>
#include <edit_gpi.h>
#include <globals.h>


ListGpis::ListGpis(RDMatrix *matrix,RDMatrix::GpioType type,
		   QWidget *parent,const char *name)
  : QDialog(parent,name,true)
{
  QString sql;
  RDSqlQuery *q;
  QListViewItem *l;
  QString list_label;

  list_matrix=matrix;
  list_type=type;
  switch(type) {
    case RDMatrix::GpioInput:
      list_tablename="GPIS";
      list_size=list_matrix->gpis();
      list_label=tr("GPI Lines");
      setCaption(tr("List GPIs"));
      break;

    case RDMatrix::GpioOutput:
      list_tablename="GPOS";
      list_size=list_matrix->gpos();
      list_label=tr("GPO Lines");
      setCaption(tr("List GPOs"));
      break;
  }

  //
  // Fix the Window Size
  //
  setMinimumWidth(sizeHint().width());
  setMaximumWidth(sizeHint().width());
  setMinimumHeight(sizeHint().height());
  setMaximumHeight(sizeHint().height());

  //
  // Create Fonts
  //
  QFont bold_font=QFont("Helvetica",12,QFont::Bold);
  bold_font.setPixelSize(12);
  QFont font=QFont("Helvetica",12,QFont::Normal);
  font.setPixelSize(12);

  //
  // Gpis List Box
  //
  list_list_view=new QListView(this,"list_box");
  list_list_view->
    setGeometry(10,24,sizeHint().width()-20,sizeHint().height()-94);
  QLabel *label=
    new QLabel(list_list_view,list_label,this,"list_list_view_label");
  label->setFont(bold_font);
  label->setGeometry(14,5,85,19);
  list_list_view->setAllColumnsShowFocus(true);
  list_list_view->setItemMargin(5);
  connect(list_list_view,
	  SIGNAL(doubleClicked(QListViewItem *,const QPoint &,int)),
	  this,
	  SLOT(doubleClickedData(QListViewItem *,const QPoint &,int)));

  switch(list_type) {
    case RDMatrix::GpioInput:
      list_list_view->addColumn(tr("GPI"));
      break;

    case RDMatrix::GpioOutput:
      list_list_view->addColumn(tr("GPO"));
      break;
  }
  list_list_view->setColumnAlignment(0,Qt::AlignHCenter);

  list_list_view->addColumn(tr("MACRO CART"));
  list_list_view->setColumnAlignment(1,Qt::AlignHCenter);

  list_list_view->addColumn(tr("DESCRIPTION"));
  list_list_view->setColumnAlignment(2,Qt::AlignLeft);

  //
  //  Edit Button
  //
  QPushButton *button=new QPushButton(this,"edit_button");
  button->setGeometry(10,sizeHint().height()-60,80,50);
  button->setFont(bold_font);
  button->setText(tr("&Edit"));
  connect(button,SIGNAL(clicked()),this,SLOT(editData()));

  //
  //  Ok Button
  //
  button=new QPushButton(this,"ok_button");
  button->setGeometry(sizeHint().width()-180,sizeHint().height()-60,80,50);
  button->setDefault(true);
  button->setFont(bold_font);
  button->setText(tr("&OK"));
  connect(button,SIGNAL(clicked()),this,SLOT(okData()));

  //
  //  Cancel Button
  //
  button=new QPushButton(this,"cancel_button");
  button->setGeometry(sizeHint().width()-90,sizeHint().height()-60,
			     80,50);
  button->setFont(bold_font);
  button->setText(tr("&Cancel"));
  connect(button,SIGNAL(clicked()),this,SLOT(cancelData()));

  //
  // Load Values
  //
  sql=QString().sprintf("select %s.NUMBER,%s.MACRO_CART,CART.TITLE \
                         from %s left join CART \
                         on %s.MACRO_CART=CART.NUMBER \
                         where (%s.STATION_NAME=\"%s\")&&(%s.MATRIX=%d)\
                         order by %s.NUMBER",
			(const char *)list_tablename,
			(const char *)list_tablename,
			(const char *)list_tablename,
			(const char *)list_tablename,
			(const char *)list_tablename,
			(const char *)list_matrix->station(),
			(const char *)list_tablename,
			list_matrix->matrix(),
			(const char *)list_tablename);
  q=new RDSqlQuery(sql);
  if(list_matrix->type()==RDMatrix::LiveWire) {
    while(q->next()) {
      l=new QListViewItem(list_list_view);
      l->setText(0,QString().sprintf("%05d",q->value(0).toInt()));
      if(q->value(1).toInt()>0) {
	l->setText(1,QString().sprintf("%06d",q->value(1).toInt()));
      }
      if(q->value(1).toInt()>0) {
	l->setText(2,q->value(2).toString());
      }
      else {
	l->setText(2,tr("[unassigned]"));
      }
    }
  }
  else {
    q->first();
    for(int i=0;i<list_size;i++) {
      l=new QListViewItem(list_list_view); 
      l->setText(0,QString().sprintf("%03d",i+1));
      if(q->isValid()&&(q->value(0).toInt()==(i+1))){
	if(q->value(1).toInt()>0) {
	  l->setText(1,QString().sprintf("%06d",q->value(1).toInt()));
	}
	if(q->value(1).toInt()>0) {
	  l->setText(2,q->value(2).toString());
	}
	else {
	  l->setText(2,tr("[unassigned]"));
	}
	q->next();
      }
      else {
	l->setText(2,tr("[unassigned]"));
      }
    }
  }
  delete q;
}


QSize ListGpis::sizeHint() const
{
  return QSize(400,250);
} 


QSizePolicy ListGpis::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
}


void ListGpis::editData()
{
  int cart=-1;

  QString desc;
  QListViewItem *item=list_list_view->selectedItem();
  if(item==NULL) {
    return;
  }
  int gpi=item->text(0).toInt();
  if(!item->text(1).isEmpty()) {
    cart=item->text(1).toInt();
  }
  EditGpi *gpi_box=new EditGpi(gpi,&cart,&desc,this,"gpi_box");
  if(gpi_box->exec()==0) {
    if(cart>0) {
      item->setText(1,QString().sprintf("%06d",cart));
      item->setText(2,desc);
    }
    else {
      item->setText(1,"");
      item->setText(2,tr("[unassigned]"));
    }
  }
  delete gpi_box;
}


void ListGpis::doubleClickedData(QListViewItem *,const QPoint &,int)
{
  editData();
}


void ListGpis::okData()
{
  QString sql;
  RDSqlQuery *q;
  RDMacro rml;

  sql=QString().sprintf("delete from %s where (STATION_NAME=\"%s\")&&\
                           MATRIX=%d",
			(const char *)list_tablename,
			(const char *)list_matrix->station(),
			list_matrix->matrix());
  q=new RDSqlQuery(sql);
  delete q;
  RDStation *station=new RDStation(list_matrix->station());
  rml.setCommand(RDMacro::GI);
  rml.setRole(RDMacro::Cmd);
  rml.setArgQuantity(5);
  rml.setArg(0,list_matrix->matrix());
  switch(list_type) {
    case RDMatrix::GpioInput:
      rml.setArg(1,"I");
      break;

    case RDMatrix::GpioOutput:
      rml.setArg(1,"O");
      break;
  }
  rml.setAddress(station->address());
  rml.setEchoRequested(false);
  delete station;

  QListViewItem *item=list_list_view->firstChild();
  while(item!=NULL) {
    sql=QString().sprintf("insert into %s set STATION_NAME=\"%s\",\
                           MATRIX=%d,NUMBER=%d,MACRO_CART=%d",
			  (const char *)list_tablename,
			  (const char *)list_matrix->station(),
			  list_matrix->matrix(),
			  item->text(0).toInt(),item->text(1).toInt());
    q=new RDSqlQuery(sql);
    delete q;
    rml.setArg(2,item->text(0).toInt());
    rml.setArg(3,true);
    rml.setArg(4,item->text(1).toInt());
    rdripc->sendRml(&rml);
    item=item->nextSibling();
  }
  done(0);
}


void ListGpis::cancelData()
{
  done(1);
}
