// post_counter.cpp
//
// The post counter widget for Rivendell
//
//   (C) Copyright 2002-2004 Fred Gleason <fredg@paravelsystems.com>
//
//      $Id: post_counter.cpp,v 1.15 2007-02-14 21:53:27 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License version 2 as
//   published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <qbitmap.h>
#include <qpainter.h>

#include <post_counter.h>
#include <colors.h>
#include <globals.h>

PostCounter::PostCounter(QWidget *parent,const char *name)
  : QPushButton(parent,name)
{
  post_running=false;
  post_time_format="hh:mm:ss";
  post_time=QTime();
  post_start=QTime();

  //
  // Generate Fonts
  //
  post_small_font=QFont("Helvetica",12,QFont::Normal);
  post_small_font.setPixelSize(12);
  post_large_font=QFont("Helvetica",26,QFont::Normal);
  post_large_font.setPixelSize(26);

  //
  // Generate Palettes
  //
  post_idle_palette=palette();
  post_early_palette=
    QPalette(QColor(POSTPOINT_EARLY_COLOR),backgroundColor());
  post_ontime_palette=
    QPalette(QColor(POSTPOINT_ONTIME_COLOR),backgroundColor());
  post_late_palette=
    QPalette(QColor(POSTPOINT_LATE_COLOR),backgroundColor());

  post_offset = 0;
  UpdateDisplay();
}

void PostCounter::setTimeMode(RDAirPlayConf::TimeMode mode)
{
  switch(mode) {
      case RDAirPlayConf::TwentyFourHour:
	post_time_format="hh:mm:ss";
	break;

      case RDAirPlayConf::TwelveHour:
	post_time_format="h:mm:ss ap";
	break;
  }
  UpdateDisplay();
}

QSize PostCounter::sizeHint() const
{
  return QSize(200,60);
}


QSizePolicy PostCounter::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
}


void PostCounter::setPostPoint(QTime point,QTime start,int offset)
{
  post_time=point;
  post_start=start;
  post_offset=offset;
  UpdateDisplay();
}


void PostCounter::tickCounter()
{
  if(post_running) {
    UpdateDisplay();
  }
}


void PostCounter::setEnabled(bool state)
{
  QWidget::setEnabled(state);
  UpdateDisplay();
}


void PostCounter::setDisabled(bool state)
{
  setEnabled(!state);
}


void PostCounter::keyPressEvent(QKeyEvent *e)
{
  e->ignore();
}


void PostCounter::UpdateDisplay()
{
  QString str;
  QString point;
  QString state;
  int msecs;
  QTime current_time=
    QTime::currentTime().addMSecs(rdstation_conf->timeOffset());
  QTime actual=post_start.addMSecs(post_offset);
  
  if(isEnabled()) {
    if(!post_time.isNull()) {
      str=QString(tr("Next Timed Start"));
      point=QString().sprintf("%s [%s]",(const char *)str,
			      (const char *)post_time.toString(post_time_format));
      if(!post_start.isNull()) {        // Running
	if(actual.addMSecs(POST_COUNTER_MARGIN)<post_time) {   // Early
	  msecs=actual.msecsTo(post_time);
	  state=QString().sprintf("-%s",
				  (const char *)QTime().addMSecs(msecs+1000).toString("hh:mm:ss"));
	  setPalette(post_early_palette);
	  post_running=false;
	}
	else {
	  if(actual>post_time.addMSecs(POST_COUNTER_MARGIN)) {  // Late
	    msecs=post_time.msecsTo(actual);
	    state=QString().sprintf("+%s",
				    (const char *)QTime().addMSecs(msecs).toString("hh:mm:ss"));
	    setPalette(post_late_palette);
	    post_running=false;
	  }
	  else {        // On Time
	    state=tr("On Time");
	    setPalette(post_ontime_palette);
	    post_running=false;
	  }
	}
      }
      else {            // Stopped
	QTime start=post_time.addMSecs(-post_offset);
	if(current_time.addMSecs(POST_COUNTER_MARGIN)<start) {   // Early
	  msecs=current_time.msecsTo(start);
	  state=QString().sprintf("-%s",
				  (const char *)QTime().addMSecs(msecs+1000).toString("hh:mm:ss"));
	  setPalette(post_early_palette);
	}
	else {
	  if(current_time>start.addMSecs(POST_COUNTER_MARGIN)) {  // Late
	    msecs=start.msecsTo(current_time);
	    state=QString().sprintf("+%s",
				    (const char *)QTime().addMSecs(msecs).toString("hh:mm:ss"));
	    setPalette(post_late_palette);
	  }
	  else {        // On Time
	    state="00:00:00";
	    setPalette(post_ontime_palette);
	  }
	}
	post_running=true;
      }
    }
    else {     // No postpoint
      point=tr("Next Timed Start [--:--:--]");
      state="--------";
      setPalette(post_idle_palette);
      post_running=false;
    }
  }
  else {       // Disabled
    point=tr("Next Timed Start [--:--:--]");
    state="--------";
    setPalette(post_idle_palette);
    post_running=false;
  }
  QBitmap pix(sizeHint().width(),sizeHint().height());
  QPainter *p=new QPainter(&pix);
  p->eraseRect(0,0,sizeHint().width(),sizeHint().height());
  p->setPen(color1);
  p->setFont(post_small_font);
  p->drawText((sizeHint().width()-p->
	       fontMetrics().width(point))/2,22,point);
  p->setFont(post_large_font);
  p->drawText((sizeHint().width()-p->
	       fontMetrics().width(state))/2,48,state);
  p->end();
  delete p;
  setPixmap(pix);    
}
